﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS.Extension.EIS;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using InfoWorld.EVS.CTSMAPI;


namespace BMS.ServicesWrapper.EIS
{
    /// <summary>
    /// Translate entity into object, translate date types into HL7 data types.
    /// </summary>
    internal static class Translator
    {
        #region translate from EIS Entity

        /// <summary>
        /// Translates the entity into ward.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        public static Ward TranslateWard(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                Ward ward = new Ward() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        ward.Name = ((ST)ivp.TraitValue).text;

                    else if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.IEN)
                                ward.Ien = ii.extension;
                        }
                    }

                    else if (ivp.TraitIdentifier == EISConstants.TYPEGROUP)
                    {
                        ward.TypeGroup = ivp.TraitValue == null ? string.Empty : ((ST)ivp.TraitValue).text;
                    }
                    else if (ivp.TraitIdentifier == EISConstants.VISTASPECIALTY)
                    {
                        string specialityCode = TranslateStringFromANY(ivp.TraitValue);
                        ward.Specialty = VistaSpecialies.FirstOrDefault(a => a.code == specialityCode);
                    }

                    else if (ivp.TraitIdentifier == EISConstants.IS_ACTIVE)
                    {
                        if (((ST)ivp.TraitValue) == null || ((ST)ivp.TraitValue).text == EISConstants.FALSE)
                            ward.IsActive = false;
                        else
                            ward.IsActive = true;
                    }
                    else if (ivp.TraitIdentifier == EISConstants.IS_CURRENTLY_OOS)
                    {
                        string value = TranslateStringFromANY(ivp.TraitValue);
                        if (string.IsNullOrEmpty(value) || value == EISConstants.TRUE)
                            ward.IsCurrentlyOOS = true;
                        else
                            ward.IsCurrentlyOOS = false;
                    }
                    else if (ivp.TraitIdentifier == EISConstants.HIDE_SPECIALTY_DESKTOP)
                    {
                        if (((ST)ivp.TraitValue) == null || ((ST)ivp.TraitValue).text == EISConstants.FALSE)
                            ward.HideSpecialtyDesktop = false;
                        else
                            ward.HideSpecialtyDesktop = true;
                    }
                    else if (ivp.TraitIdentifier == EISConstants.HIDE_SPECIALTY_KIOSK)
                    {
                        if (((ST)ivp.TraitValue) == null || ((ST)ivp.TraitValue).text == EISConstants.FALSE)
                            ward.HideSpecialtyKiosk = false;
                        else
                            ward.HideSpecialtyKiosk = true;
                    }
                    else if (ivp.TraitIdentifier == EISConstants.IS_VIRTUAL_WARD)
                    {
                        if (((ST)ivp.TraitValue) == null || ((ST)ivp.TraitValue).text == EISConstants.FALSE)
                            ward.IsVirtual = false;
                        else
                            ward.IsVirtual = true;
                    }
                    else if (ivp.TraitIdentifier == EISConstants.CENSUS_CATEGORY)
                    {
                        string censusCategoryCode = TranslateStringFromANY(ivp.TraitValue);
                        ward.CensusCategory = CensusCategories.FirstOrDefault(a => a.code == censusCategoryCode);
                    }
                    else if (ivp.TraitIdentifier == EISConstants.ASSIGNED_VIRTUAL_WARD)
                    {
                        string virtualWardId = TranslateStringFromANY(ivp.TraitValue);

                        if (!string.IsNullOrEmpty(virtualWardId))
                            ward.AssignedVirtualWard = new Ward() { Id = new II (Constants.ADMISSIONROOT, virtualWardId) };
                        else
                            ward.AssignedVirtualWard = null;
                    }
                }

                //get parents (diviion list);
                List<II> divisions = TraitManager.FindScopers(entity, EISConstants.SDLOC);
                if (divisions != null)
                {
                    ward.DivisionList = new List<Division>();
                    foreach (II id in divisions)
                        ward.DivisionList.Add(new Division() { Id = id });
                }

                II parentVistaSiteId = TraitManager.FindScoper(entity, EISConstants.OWN);
                if (parentVistaSiteId != null)
                    ward.VistaSite = new VistaSite() { Id = parentVistaSiteId };

                return ward;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Translates the entity into division.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        public static Division TranslateDivision(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                Division division = new Division() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        division.Name = ((ST)ivp.TraitValue).text;

                    if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.IEN)
                                division.Ien = ii.extension;
                        }
                    }

                    if (ivp.TraitIdentifier == EISConstants.WHITEBOARD_REPORT_PATH)
                        division.WhiteboardReportPath = (ivp.TraitValue != null) ? ((ST)ivp.TraitValue).text : null;
                }

                //get parent (facility);
                II facilityId = TraitManager.FindScoper(entity, EISConstants.MBR);
                if (facilityId != null)
                    division.Facility = new Facility() { Id = facilityId };

                return division;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Translates the facility from entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        public static Facility TranslateFacility(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                Facility facility = new Facility() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        facility.Name = ((ST)ivp.TraitValue).text;

                    if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.CODE)
                                facility.Code = ii.extension;
                        }
                    }

                    if (ivp.TraitIdentifier == EISConstants.IS_ACTIVE)
                    {
                        if (((ST)ivp.TraitValue) == null || ((ST)ivp.TraitValue).text == EISConstants.FALSE)
                            facility.IsActive = false;
                        else
                            facility.IsActive = true;
                    }

                    if (ivp.TraitIdentifier == EISConstants.ADDRESS1)
                        facility.Address1 = ivp.TraitValue as AD;

                    if (ivp.TraitIdentifier == EISConstants.ADDRESS2)
                        facility.Address2 = ivp.TraitValue as AD;

                    if (ivp.TraitIdentifier == EISConstants.PHONE)
                    {
                        TEL tel = ivp.TraitValue as TEL;
                        if (tel != null)
                            facility.Telephone = tel.value.Substring("tel:".Length);
                    }

                    if (ivp.TraitIdentifier == EISConstants.PERSON_CONTACT_NAME)
                        facility.PersonContactName = (ST)ivp.TraitValue;

                    if (ivp.TraitIdentifier == EISConstants.EMAIL)
                    {
                        TEL tel = ivp.TraitValue as TEL;
                        if (tel != null)
                            tel.value = tel.value.Substring(EISConstants.EMAIL_CONTACT_VALUE.Length);
                        facility.PersonContactEmail = tel;
                    }

                    if (ivp.TraitIdentifier == EISConstants.TELEPHONE)
                    {
                        TEL tel = ivp.TraitValue as TEL;
                        if (tel != null)
                            tel.value = tel.value.Substring(EISConstants.TELEPHONE_CONTACT_VALUE.Length);
                        facility.PersonContactPhone = ivp.TraitValue as TEL;
                    }

                    if (ivp.TraitIdentifier == EISConstants.IS_ALIVE)
                    {
                        if (((ST)ivp.TraitValue) == null || ((ST)ivp.TraitValue).text == EISConstants.FALSE)
                            facility.IsAlive = false;
                        else
                            facility.IsAlive = true;
                    }

                    if (ivp.TraitIdentifier == EISConstants.SITE_NUMBER)
                        facility.SiteNumber = ((ST)ivp.TraitValue).text;
                }

                //get parent (vista site)
                II vistaSiteId = TraitManager.FindScoper(entity, EISConstants.MBR);
                if (vistaSiteId != null)
                    facility.VistaSite = new VistaSite() { Id = vistaSiteId };

                return facility;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Translates the hospital location from entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        public static HospitalLocation TranslateHospitalLocation(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                HospitalLocation hospitalLocation = new HospitalLocation() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        hospitalLocation.Name = ((ST)ivp.TraitValue).text;
                    else if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.IEN)
                                hospitalLocation.Ien = ii.extension;
                            if (ii.root == EISConstants.CODE)
                                hospitalLocation.Abbrevation = ii.extension;
                        }
                    }
                    else if (ivp.TraitIdentifier == EISConstants.PHONE)
                    {
                        TEL tel = ivp.TraitValue as TEL;
                        if (tel != null)
                            hospitalLocation.Telephone = tel.value.Substring("tel:".Length);
                    }
                    else if (ivp.TraitIdentifier == EISConstants.WARD_LOCATION_ID)
                    {
                        hospitalLocation.WardLocationId = TranslateStringFromANY(ivp.TraitValue);
                    }
                }

                //get parent (vista site)
                II vistaSiteId = TraitManager.FindScoper(entity, EISConstants.MBR);
                if (vistaSiteId != null)
                    hospitalLocation.VistaSite = new VistaSite() { Id = vistaSiteId };

                return hospitalLocation;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static Visn TranslateVisn(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                Visn visn = new Visn() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        visn.Name = ((ST)ivp.TraitValue).text;

                    if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.NUMBER)
                                visn.Number = ii.extension;
                        }
                    }
                }
                //get parent (region)
                II regionId = TraitManager.FindScoper(entity, EISConstants.MBR);
                if (regionId != null)
                    visn.Region = new Region() { Id = regionId };
                return visn;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static VistaSite TranslateVistaSite(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                VistaSite vistaSite = new VistaSite() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        vistaSite.Name = ((ST)ivp.TraitValue).text;

                    if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.CODE)
                                vistaSite.Code = ii.extension;

                            if (ii.root == EISConstants.NUMBER)
                                vistaSite.Number = ii.extension;
                        }
                    }
                }
                //get parent (visn)
                II visnId = TraitManager.FindScoper(entity, EISConstants.MBR);
                if (visnId != null)
                    vistaSite.Visn = new Visn() { Id = visnId };
                return vistaSite;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static Region TranslateRegion(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                Region region = new Region() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        region.Name = ((ST)ivp.TraitValue).text;

                    if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.NUMBER)
                            {
                                region.Number = Convert.ToInt32(ii.extension);
                            }
                        }
                    }
                }
                return region;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static Bed TranslateBed(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity == null)
                    return null;

                Bed bed = new Bed() { Id = ExtractIdentifier(entity) };
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME)
                        bed.Name = ((ST)ivp.TraitValue).text;
                    if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.IEN)
                                bed.Ien = ii.extension;
                            if (ii.root == EISConstants.SITE)
                                bed.VistaSite = new VistaSite() { Id = new II(bed.Id.root, ii.extension) };
                        }
                    }
                    ST st = null;
                    st = ivp.TraitValue as ST;
                    if (st != null)
                    {
                        if (ivp.TraitIdentifier == EISConstants.NURSE_ASSIGNMENT && ivp.TraitValue != null)
                            bed.NurseAssignment = st.text;
                        if (ivp.TraitIdentifier == EISConstants.REASON && ivp.TraitValue != null)
                            bed.Reason = new CD() { code = st.text };
                        if (ivp.TraitIdentifier == EISConstants.BED_STATUS && ivp.TraitValue != null)
                            bed.BedStatus = new CD() { code = st.text };
                    }
                    if (ivp.TraitIdentifier == EISConstants.IS_BED_HOLD && ivp.TraitValue != null)
                    {
                        if (((ST)ivp.TraitValue) == null || ((ST)ivp.TraitValue).text == EISConstants.FALSE)
                            bed.IsBedHold = false;
                        else
                            bed.IsBedHold = true;
                    }
                }

                //get parent (ward list);
                List<II> wards = TraitManager.FindScopers(entity, EISConstants.LOCE);
                if (wards != null)
                {
                    bed.WardList = new List<Ward>();
                    foreach (II id in wards)
                        bed.WardList.Add(new Ward() { Id = id });
                }
                // get grandfathers(division list) and comments
                //the comments are saved on the MNT role as NameFamily field
                BedComment bc = null;
                bed.CommentList = new List<BedComment>();
                foreach (IdentifierValuePair identifierValuePair in entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.MNT))
                {
                    R_MaintainedEntity role = identifierValuePair.TraitValue as R_MaintainedEntity;
                    if (role != null && role.Scoper != null && role.Scoper.Identifier != null)
                    {
                        bc = new BedComment() { Division = new Division() { Id = new II() { root = role.Scoper.DomainIdentifier, extension = role.Scoper.Identifier } }, Comment = (role.NameFamily != null) ? role.NameFamily.text : string.Empty };
                        bed.CommentList.Add(bc);
                    }
                }
                return bed;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static Person TranslatePerson(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Person person = new Person();
                foreach (IdentifierValuePair ivp in entity.TraitList)
                {
                    if (ivp.TraitIdentifier == EISConstants.NAME_FAMILY)
                        person.LastName = TranslateStringFromANY(ivp.TraitValue);
                    else if (ivp.TraitIdentifier == EISConstants.NAME_FIRST_GIVEN)
                        person.FirstName = TranslateStringFromANY(ivp.TraitValue);
                    else if (ivp.TraitIdentifier == EISConstants.NAME_SECOND_GIVEN)
                        person.MiddleName = TranslateStringFromANY(ivp.TraitValue);
                    else if (ivp.TraitIdentifier == EISConstants.GENDER)
                        person.Gender = ivp.TraitValue as CD;
                    else if (ivp.TraitIdentifier == EISConstants.UID)
                    {
                        II ii = null;
                        ii = ivp.TraitValue as II;
                        if (ii != null)
                        {
                            if (ii.root == EISConstants.IEN)
                                person.Ien = ii.extension;
                            else if (ii.root == EISConstants.SSN)
                                person.SSN = ii;
                            else if (ii.root == EISConstants.SITE)
                                person.VistaSite = ii;
                        }
                    }
                    else if (ivp.TraitIdentifier == EISConstants.DATE_OF_BIRTH)
                    {
                        ANY x = null;
                        x = ivp.TraitValue;
                        if (x != null)
                        {
                            TS ts = x as TS;
                            if (ts != null)
                            {
                                person.DateOfBirth = (ts.dateTimeValue.HasValue) ? ts.dateTimeValue.Value.ToUniversalTime() : new Nullable<DateTime>();
                            }
                        }
                    }
                    else if (ivp.TraitIdentifier == EISConstants.PERSON_WEIGHT)
                        person.Weight = TranslateStringFromANY(ivp.TraitValue);
                }
                person.Id = ExtractIdentifier(entity);

                return person;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static Patient TranslatePatient(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Person person = TranslatePerson(entity);
                if (person == null)
                    return null;
                else
                {
                    Patient patient = new Patient()
                    {
                        FirstName = person.FirstName,
                        Gender = person.Gender,
                        Id = person.Id,
                        LastName = person.LastName,
                        MiddleName = person.MiddleName,
                        SSN = person.SSN,
                        DateOfBirth = person.DateOfBirth,
                        Weight = person.Weight

                    };
                    IdentifierValuePair treatingSpecialty = entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.TREATING_SPECIALTY).FirstOrDefault();
                    if (treatingSpecialty != null)
                    {
                        string specialityCode = TranslateStringFromANY(treatingSpecialty.TraitValue);
                        if (!string.IsNullOrWhiteSpace(specialityCode))
                            patient.TreatingSpecialty = TreatingSpecialties.Where(a => a.code == specialityCode).FirstOrDefault();
                    }

                    IdentifierValuePair isVistaValidated = entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.IS_VISTA_VALIDATED).FirstOrDefault();
                    if (isVistaValidated != null)
                    {
                        string value = TranslateStringFromANY(isVistaValidated.TraitValue);
                        if (string.IsNullOrEmpty(value) || value == EISConstants.TRUE)
                            patient.IsVistaValidated = true;
                        else
                            patient.IsVistaValidated = false;
                    }
                    else
                        patient.IsVistaValidated = true;

                    IdentifierValuePair isPatientOptOut = entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.IS_PATIENT_OPTOUT).FirstOrDefault();
                    if (isPatientOptOut != null)
                    {
                        string value = TranslateStringFromANY(isPatientOptOut.TraitValue);
                        if (string.IsNullOrEmpty(value) || value == "1")
                            patient.IsOptOut = true;
                        else
                            patient.IsOptOut = false;
                    }
                    else
                        patient.IsVistaValidated = true;

                    IdentifierValuePair admittingDiagnosis = entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.ADMITTING_DIAGNOSIS).FirstOrDefault();
                    if (admittingDiagnosis != null)
                        patient.AdmittingDiagnosis = TranslateStringFromANY(admittingDiagnosis.TraitValue);

                    IdentifierValuePair serviceConnectedPercentageIdent = entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.SERVICE_CONNECTED_PERCENTAGE).FirstOrDefault();
                    if (serviceConnectedPercentageIdent != null)
                    {
                        ST stValue = serviceConnectedPercentageIdent.TraitValue as ST;
                        patient.ServiceConnectedPercentage = stValue == null ? "" : stValue.text ?? "";
                    }
                    IdentifierValuePair attendingPhysician = entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.ATTENDING_PHYSICIAN).FirstOrDefault();
                    if (attendingPhysician != null && attendingPhysician.TraitValue != null)
                    {
                        ST stValue = attendingPhysician.TraitValue as ST;
                        if (stValue != null && !string.IsNullOrEmpty(stValue.text))
                        {
                            int index = stValue.text.IndexOf('_');
                            string root = stValue.text.Substring(0, index);
                            string extension = stValue.text.Substring(index + 1, stValue.text.Length - index - 1);
                            patient.AttendingPhysicianId = new II(root, extension);
                        }
                    }

                    IdentifierValuePair listIen = entity.TraitList.Where(a => a.TraitIdentifier == EISConstants.IEN_LIST).FirstOrDefault();
                    if (listIen != null && listIen.TraitValue != null)
                    {
                        ST stValue = listIen.TraitValue as ST;
                        string value = "";
                        if (stValue != null)
                        {
                            value = stValue.text ?? "";
                        }

                        if (!string.IsNullOrEmpty(value))
                        {
                            string[] ienValues = value.Split(new char[] { '|' }, StringSplitOptions.RemoveEmptyEntries);
                            patient.IENList = new List<II>();
                            string[] parts = null;
                            foreach (string vistaCodeIen in ienValues)
                            {
                                parts = vistaCodeIen.Split('_');
                                patient.IENList.Add(new II { root = parts[0], extension = parts[1] });
                            }
                        }
                    }

                    return patient;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        /// <summary>
        /// Extracts the internal identifier of the entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        public static II ExtractIdentifier(Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return entity == null ? null : new II() { root = entity.DomainIdentifier, extension = entity.Identifier };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Translates the string from ANY.
        /// </summary>
        /// <param name="value">The value.</param>
        /// <returns></returns>
        public static string TranslateStringFromANY(ANY value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (value == null)
                    return null;

                if (value is BIN)
                {
                    BIN x = value as BIN;
                    if (x != null)
                    {
                        return x.text;
                    }
                }

                if (value is CD)
                {
                    CD x = value as CD;
                    if (x != null)
                    {
                        return x.code;
                    }
                }

                if (value is II)
                {
                    II x = value as II;
                    if (x != null)
                    {
                        return x.extension;
                    }
                }

                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static CE TranslateCEFromCD(CD cd)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new CE() { code = cd.code, codeSystem = cd.codeSystem, codeSystemName = cd.codeSystemName, displayName = cd.displayName };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region translate to EIS Entity

        internal static void TranslateToEntity(Place place, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                entity.TraitList.Add(TraitManager.CreateIdentifierForName(place.Name));
                entity.TraitList.Add(TraitManager.CreateIdentifierForIen(place.Ien));
                entity.TraitList.Add(TraitManager.CreateIdentifierForPlaceType(place.GetPlaceType()));
                if (place.Address1 != null)
                    entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.ADDRESS1, place.Address1));
                if (place.Address2 != null)
                    entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.ADDRESS2, place.Address2));
                if (place.GetPlaceType() != EISConstants.IDENTIFIER_FOR_WARD_CODE)
                    entity.TraitList.Add(TraitManager.CreateIdentifierForIsActive(true));
                if (place.PersonContactName != null)
                    entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.PERSON_CONTACT_NAME, place.PersonContactName));
                if (place.PersonContactEmail != null)
                    entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.EMAIL, place.PersonContactEmail, ContactType.Email));
                if (place.PersonContactPhone != null)
                    entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.TELEPHONE, place.PersonContactPhone, ContactType.Phone));

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Region region, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)region, entity);                
                entity.TraitList.Add(TraitManager.CreateIdentifierForNumber(region.Number.ToString()));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Visn visn, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)visn, entity);                
                entity.TraitList.Add(TraitManager.CreateIdentifierForNumber(visn.Number));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(VistaSite vistaSite, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)vistaSite, entity);                
                entity.TraitList.Add(TraitManager.CreateIdentifierForCode(vistaSite.Code));
                entity.TraitList.Add(TraitManager.CreateIdentifierForNumber(vistaSite.Number));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(HospitalLocation hospitalLocation, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)hospitalLocation, entity);                
                entity.TraitList.Add(TraitManager.CreateIdentifierForCode(hospitalLocation.Abbrevation));
                IdentifierValuePair ivp = TraitManager.CreateIdentifierForPhone(hospitalLocation.Telephone);
                if (string.IsNullOrEmpty(hospitalLocation.Telephone))
                    ivp.action = InfoWorld.HL7.ITS.Extension.EIS.Action.DELETE;
                entity.TraitList.Add(ivp);                
                entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.WARD_LOCATION_ID, hospitalLocation.WardLocationId));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Facility facility, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)facility, entity);                
                entity.TraitList.Add(TraitManager.CreateIdentifierForCode(facility.Code));
                entity.TraitList.Add(TraitManager.CreateIdentifierForPhone(facility.Telephone));
                entity.TraitList.Add(TraitManager.CreateIdentifierForIsAlive(facility.IsAlive));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Division division, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)division, entity);                
                entity.TraitList.Add(new IdentifierValuePair(EISConstants.WHITEBOARD_REPORT_PATH, new ST(division.WhiteboardReportPath)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Ward ward, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)ward, entity);                
                entity.TraitList.Add(TraitManager.CreateIdentifierForIsActive(ward.IsActive));
                entity.TraitList.Add(TraitManager.CreateIdentifierForVistaSpecialty(ward.Specialty == null ? string.Empty : ward.Specialty.code));
                entity.TraitList.Add(TraitManager.CreateIdentifierForTypeGroup(ward.TypeGroup));
                entity.TraitList.Add(TraitManager.CreateIdentifierForIsCurrentlyOOS(ward.IsCurrentlyOOS));

                entity.TraitList.Add(TraitManager.CreateIdentifierForHideSpecialtyDesktop(ward.HideSpecialtyDesktop));
                entity.TraitList.Add(TraitManager.CreateIdentifierForHideSpecialtyKiosk(ward.HideSpecialtyKiosk));
                entity.TraitList.Add(TraitManager.CreateIdentifierForIsVirtualWard(ward.IsVirtual));

                entity.TraitList.Add(TraitManager.CreateIdentifierForCensusCategory(ward.CensusCategory == null ? string.Empty : ward.CensusCategory.code));
                entity.TraitList.Add(TraitManager.CreateIdentifierForAssignedVirtualWard(ward.AssignedVirtualWard == null ? string.Empty : ward.AssignedVirtualWard.Id.extension));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Bed bed, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslateToEntity((Place)bed, entity);                
                entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.NURSE_ASSIGNMENT, bed.NurseAssignment));
                entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.REASON, bed.Reason));
                entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.BED_STATUS, bed.BedStatus));
                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.IS_BED_HOLD, TraitValue = (bed.IsBedHold) ? new ST(EISConstants.TRUE) : new ST(EISConstants.FALSE) });
                if (bed.VistaSite != null)
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.UID, TraitValue = new II() { root = EISConstants.SITE, extension = bed.VistaSite.Id.extension } });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Person person, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslatePersonToEntity(person, entity);

                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.PROV, TraitValue = new R_HealthCareProvider() { Player = person.GetEntityInRole(), Scoper = null }, });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static void TranslateToEntity(Patient patient, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TranslatePersonToEntity((Person)patient, entity);

                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.PAT, TraitValue = new R_Patient() { Player = patient.GetEntityInRole(), Scoper = null }, });
                if (patient.TreatingSpecialty != null && !string.IsNullOrWhiteSpace(patient.TreatingSpecialty.code))
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.TREATING_SPECIALTY, TraitValue = new ST(patient.TreatingSpecialty.code) });
                else
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.TREATING_SPECIALTY, TraitValue = new ST() });
                entity.TraitList.Add(TraitManager.CreateIdentifierForIsVistaValidated(patient.IsVistaValidated));
                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.ADMITTING_DIAGNOSIS, TraitValue = new ST(patient.AdmittingDiagnosis) });
                entity.TraitList.Add(TraitManager.CreateIdentifierForIsPatientOptOut(patient.IsOptOut));

                string serviceConnectedPercentageValue = patient.ServiceConnectedPercentage ?? "";
                entity.TraitList.Add(TraitManager.CreateIdentifier(EISConstants.SERVICE_CONNECTED_PERCENTAGE, serviceConnectedPercentageValue));
                ST attendingPhysician = patient.AttendingPhysicianId == null ? new ST() : new ST(string.Format("{0}_{1}", patient.AttendingPhysicianId.root, patient.AttendingPhysicianId.extension));
                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.ATTENDING_PHYSICIAN, TraitValue = attendingPhysician });

                StringBuilder ien_list = new StringBuilder();
                if (patient.IENList != null && patient.IENList.Count > 0)
                {
                    foreach (II patientIen in patient.IENList)
                    {
                        ien_list.Append(patientIen.root);
                        ien_list.Append("_");
                        ien_list.Append(patientIen.extension);
                        ien_list.Append("|");
                    }
                    if (ien_list.Length > 0)
                        ien_list = ien_list.Remove(ien_list.Length - 1, 1);
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.IEN_LIST, TraitValue = new ST(ien_list.ToString()) });
                }
                else
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.IEN_LIST, TraitValue = new ST() });

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static void TranslatePersonToEntity(Person person, Entity entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.NAME_FAMILY, TraitValue = new ST(person.LastName) });
                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.NAME_FIRST_GIVEN, TraitValue = new ST(person.FirstName) });
                entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.NAME_SECOND_GIVEN, TraitValue = new ST(person.MiddleName) });
                if (person.Gender != null)
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.GENDER, TraitValue = Translator.TranslateCEFromCD(person.Gender) });
                if (person.SSN != null)
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.UID, TraitValue = person.SSN });
                if (person.Ien != null)
                    entity.TraitList.Add(TraitManager.CreateIdentifierForIen(person.Ien));
                if (person.VistaSite != null)
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.UID, TraitValue = new II() { root = EISConstants.SITE, extension = person.VistaSite.extension } });
                if (person.DateOfBirth != null)
                    entity.TraitList.Add(new IdentifierValuePair() { TraitIdentifier = EISConstants.DATE_OF_BIRTH, TraitValue = TranslateAndIgnoreTimeZoneForDateOfBirth(person.DateOfBirth.Value) });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        private static ANY TranslateAndIgnoreTimeZoneForDateOfBirth(DateTime dateTime)
        {
            TS ts = new TS();
            ts.dateTimeValue = new DateTime(dateTime.Year, dateTime.Month, dateTime.Day, 0, 0, 0, DateTimeKind.Utc);

            return ts;
        }

        private static IEnumerable<CD> VistaSpecialies
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    EVS.CodeFilterParameters codeFilterParam = new EVS.CodeFilterParameters();
                    codeFilterParam.VocabularyDomain = Util.Vocabulary.VistaSpecialty.ToString();
                    codeFilterParam.MaxSelectedCodes = int.MaxValue;
                    return BMS.ServicesWrapper.EVS.EVSFactory.InstanceFromWCF.GetCodes(codeFilterParam);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        private static IEnumerable<CD> TreatingSpecialties
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    EVS.CodeFilterParameters codeFilterParam = new EVS.CodeFilterParameters();
                    codeFilterParam.VocabularyDomain = Util.Vocabulary.VistaTreatingSpecialty.ToString();
                    codeFilterParam.MaxSelectedCodes = int.MaxValue;
                    return BMS.ServicesWrapper.EVS.EVSFactory.InstanceFromWCF.GetCodes(codeFilterParam);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        private static IEnumerable<CD> CensusCategories
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    EVS.CodeFilterParameters codeFilterParam = new EVS.CodeFilterParameters();
                    codeFilterParam.VocabularyDomain = Util.Vocabulary.CensusCategory.ToString();
                    codeFilterParam.MaxSelectedCodes = int.MaxValue;
                    return BMS.ServicesWrapper.EVS.EVSFactory.InstanceFromWCF.GetCodes(codeFilterParam);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }
    }
}
